import React, { Fragment } from 'react';
import PropTypes from 'prop-types';
import { Divider, message, Form, Popconfirm } from 'antd';
import moment from 'moment';
import router from 'umi/router';
import ExportInfo from '../Export/index';
import { text, number, date, getFormArrayConfig } from '../config/index';
import ButtonDiy from '@/baseComponent/ButtonDiy';
import Shell from '../Shell';
import Edit from '../Edit/index';
import ImportUtil from '../ImportUtil/ImportUtil';
import SearchDom from '@/highOrderComponent/SearchDom';
import StandardTable from '@/components/StandardTable';
import AddOrEditModal from './AddOrEditModal';
import {
	fetchTableHeader,
	fetchTableData,
	addOrEditTableItem,
	deleteTableItem,
} from '@/webPublic/Services';
import { ModalConfirm } from '@/baseComponent/Modal';

const dateRender = (val) => val && moment(val).format('YYYY-MM-DD HH:mm:ss');

const getValue = (obj) =>
	Object.keys(obj)
		.map((key) => obj[key])
		.join(',');

/**
 * 展示list数据，
 * 根据元数据id显示元数据表格，根据模板id新增或编辑某条数据
 */
class ListWithAddEditTemplate extends React.Component {
	static propTypes = {
		objId: PropTypes.string.isRequired, // 元数据表id
		templateCode: PropTypes.string.isRequired, // 表单模板id
		filters: PropTypes.object, // 请求元数据表格时额外的过滤条件
		hasImport: PropTypes.bool, // 是否有导入操作按钮，默认false
		hasExport: PropTypes.bool, // 是否有导出按钮，默认false
	};

	static defaultProps = {
		filters: {},
		hasImport: false,
		hasExport: false,
	};

	constructor(props) {
		super(props);

		this.state = {
			modalVisible: false,
			formData: null, // 编辑某行数据时表单的数据
			headerList: [], // 表头的数据
			searchConfig: [],
			formValues: {}, // 搜索区域的数据
			isAdd: false, // 是新增还是编辑
			selectedRows: [],
			primaryKey: null,
			// showEdit: false, // 是否显示编辑区域
			pageData: {
				list: [],
				pagination: {},
			},
			confirmLoading: false,
		};
		this.columns = [];
	}

	// 渲染值
	componentDidMount() {
		const { objId } = this.props;
		fetchTableHeader(objId).then((headerList) => {
			if (!headerList) return;
			this.setState({ headerList });

			this.columns = headerList
				.filter((i) => !i.isHidden)
				.slice(0, 10)
				.map((item) => ({
					title: item.title,
					dataIndex: item.name,
					render: date.includes(item.type) && dateRender,
				}));
			const operation = {
				title: '操作',
				fixed: 'right',
				width: 120,
				render: (_, record) => {
					return (
						<Fragment>
							<a onClick={this.modify.bind(this, record)}>编辑</a>
							<Divider type="vertical" />
							<a onClick={this.delete.bind(this, record)}>删除</a>
						</Fragment>
					);
				},
			};
			this.columns.push(operation);

			const primaryKey = (headerList.find((i) => i.isPrimaryKey) || {}).name;

			this.setState({ primaryKey });

			const searchConfig = headerList.filter((i) => i.isShowQuery).map((item) => {
				const sear = getFormArrayConfig([item]);
				const { required, placeholder, ...config } = sear[0];
				return config;
			});
			this.setState({ searchConfig, primaryKey });
			this.getPage();
		});
	}

	// 新增按钮事件
	add = () => {
		const { headerList } = this.state;
		// const showEdit = headerList.length >= 10;
		this.setState({
			modalVisible: true,
			formData: {},
			isAdd: true,
			// showEdit,
		});
	};

	// 修改按钮事件
	modify = (record) => {
		const recordTemp = { ...record, update_time: Date.now() };
		const { headerList } = this.state;
		// const showEdit = headerList.length >= 10;
		// 把record里明文key改成转换后的key，用于在模板表单中回显
		const headerMap = headerList.reduce((acc, header) => ({ ...acc, [header.name]: header }), {});
		const formData = Object.keys(recordTemp).reduce((acc, key) => {
			if (!headerMap[key]) return acc;
			return { ...acc, [headerMap[key].base52]: recordTemp[key] };
		}, {});
		this.setState({
			formData,
			modalVisible: true,
			isAdd: false,
			// showEdit,
		});
	};

	delete = (record) => {
		ModalConfirm('确定删除？', {
			onOk: () => {
				const { objId } = this.props;
				// 找出主键
				const { primaryKey } = this.state;
				const Keys = {};

				Keys[primaryKey] = record[primaryKey];
				deleteTableItem(objId, primaryKey, record[primaryKey]).then((res) => {
					this.getPage();
				});
			},
		});
	};

	changeFormData = (value, key) => {
		this.setState(({ formData }) => ({ formData: { ...formData, [key]: value } }));
	};

	getPage = () => {
		const { objId, filters = {} } = this.props;
		const {
			formValues,
			pageData: { pagination },
		} = this.state;
		const query = { ...formValues, ...filters };
		const pageNo = pagination.current ? pagination.current : 1;
		const pageSize = pagination.pageSize ? pagination.pageSize : 10;

		fetchTableData({ dataObjId: objId, query, pageSize, pageNo }).then((res) => {
			if (!res.rows) {
				console.log('分页接口rows返回null报错');
			}
			this.setState({
				pageData: {
					list: res.rows,
					pagination: {
						current: res.pageNo,
						pageSize: res.pageSize,
						total: Number(res.total),
					},
				},
			});
		});
	};

	handleStandardTableChange = (pagination, filtersArg, sorter) => {
		this.setState(
			({ pageData }) => ({
				pageData: { ...pageData, pagination },
			}),
			this.getPage,
		);
	};

	handleAdd = () => {
		const {
			objId,
			form: { validateFields },
		} = this.props;
		const { isAdd } = this.state;
		validateFields((err, values) => {
			if (err) return;
			this.setState({ confirmLoading: true });
			addOrEditTableItem({ objId, data: values, isAdd }).then((res) => {
				this.setState({ confirmLoading: false });
				if (res === true) {
					message.success('操作成功');
					this.setState({
						modalVisible: false,
					});
					this.getPage();
				} else {
					message.error('操作失败');
				}
			});
		});
	};

	handleModalVisible = () => {
		this.setState({
			modalVisible: this.state.modalVisible ? false : true,
			formData: {},
		});
	};

	handleSelectRows = (rows) => {
		this.setState({
			selectedRows: rows,
		});
	};

	batchDelete = (e) => {
		const { selectedRows, primaryKey } = this.state;
		const { objId } = this.props;

		if (!selectedRows) return;

		ModalConfirm('确定删除？', {
			onOk: () => {
				deleteTableItem(objId, primaryKey, selectedRows.map((row) => row[primaryKey])).then(
					(res) => {
						this.setState({
							selectedRows: [],
						});
						this.getPage();
					},
				);
			},
		});
	};

	goBack = () => {
		router.goBack();
	};

	formStateChange = (value, key) => {
		this.setState(({ formValues }) => ({ formValues: { ...formValues, [key]: value } }));
	};

	resetFormValues = () => {
		this.setState({
			formValues: {},
		});
	};

	render() {
		const {
			modalVisible,
			selectedRows,
			isAdd,
			objId,
			pageData,
			showEdit,
			formData,
			primaryKey,
			formValues,
			searchConfig,
			confirmLoading,
		} = this.state;
		const { hasExport, form, hasImport, templateCode } = this.props;

		const addOrEditModalProps = {
			handleAdd: this.handleAdd,
			handleModalVisible: this.handleModalVisible,
			content: formData,
			form,
			isAdd,
			templateCode,
			modalVisible,
			confirmLoading,
		};
		return (
			<>
				{searchConfig.length > 0 ? (
					<SearchDom
						formStateChange={this.formStateChange}
						formValues={formValues}
						getPage={this.getPage}
						resetFormValues={this.resetFormValues}
						config={{ condition: searchConfig, fromTab: true }}
					/>
				) : null}
				<Shell styleShell={{ marginTop: searchConfig && searchConfig.length ? 16 : 0 }}>
					<div style={{ paddingLeft: '10px', marginTop: '20px' }}>
						<ButtonDiy icon="plus" className="primaryBlue" handleClick={this.add} name="新建" />
						{hasImport && <ImportUtil objId={objId} />}
						{hasExport && <ExportInfo objId={objId} />}
						<ButtonDiy handleClick={this.batchDelete} name="批量删除" />
					</div>
					<StandardTable
						rowKey="id"
						data={pageData}
						columns={this.columns}
						selectedRows={selectedRows}
						onSelectRow={this.handleSelectRows}
						onChange={this.handleStandardTableChange}
						scroll={{ x: true }}
					/>
				</Shell>

				<AddOrEditModal {...addOrEditModalProps} />
			</>
		);
	}
}

export default Form.create()(ListWithAddEditTemplate);
