/** Used to stand-in for `undefined` hash values. */
/** 用于代替“未定义的”散列值。 */
const HASH_UNDEFINED = '__lodash_hash_undefined__';

class Hash {
	/**
	 * Creates a hash object.
	 * 创建一个散列对象。
	 * @private
	 * @constructor
	 * @param {Array} [entries] 要缓存的键值对.
	 */
	constructor(entries) {
		let index = -1;
		const length = entries == null ? 0 : entries.length;

		this.clear();
		while (++index < length) {
			const entry = entries[index];
			this.set(entry[0], entry[1]);
		}
	}

	/**
	 * Removes all key-value entries from the hash.
	 * 从散列中删除所有键值项。
	 * @memberOf Hash
	 */
	clear() {
		this.__data__ = Object.create(null);
		this.size = 0;
	}

	/**
	 * Removes `key` and its value from the hash.
	 * 从散列中删除' key '及其值。
	 * @memberOf Hash
	 * @param {Object} hash 要修改的散列。
	 * @param {string} key 要删除的值的键。
	 * @returns {boolean} 如果条目被删除，则返回“true”，否则返回“false”
	 */
	delete(key) {
		const result = this.has(key) && delete this.__data__[key];
		this.size -= result ? 1 : 0;
		return result;
	}

	/**
	 * Gets the hash value for `key`.
	 * 获取' key '的散列值。
	 * @memberOf Hash
	 * @param {string} key 要获取的值的键。
	 * @returns {*} 返回条目值。
	 */
	get(key) {
		const data = this.__data__;
		const result = data[key];
		return result === HASH_UNDEFINED ? undefined : result;
	}

	/**
	 * Checks if a hash value for `key` exists.
	 * 检查' key '的哈希值是否存在。
	 * @memberOf Hash
	 * @param {string} key 输入键值检查。
	 * @returns {boolean} 如果' key '的条目存在，则返回' true '，否则返回' false '。
	 */
	has(key) {
		const data = this.__data__;
		return data[key] !== undefined;
	}

	/**
	 * Sets the hash `key` to `value`.
	 * 将散列“key”设置为“value”。
	 * @memberOf Hash
	 * @param {string} key 要设置的值的键.
	 * @param {*} value 要设置的值。
	 * @returns {Object} 返回哈希实例。
	 */
	set(key, value) {
		const data = this.__data__;
		this.size += this.has(key) ? 0 : 1;
		data[key] = value === undefined ? HASH_UNDEFINED : value;
		return this;
	}
}

export default Hash;
