import React from 'react';
import JsBarcode from 'jsbarcode';
import styles from '../index.less';
import moment from 'moment';
import { styleCount } from './config';
import { isJSON } from '@/webPublic/one_stop_public/copy';

/**
 * -2 条形码
 * -1 常量
 * 0  文字
 * 1 时间
 * 2 照片
 * */

const normalTextRender = (text, config) => {
	if (!text || text === 'undefined' || typeof text === 'undefined') {
		return null;
	}
	const configInfoJSON = (config.info && isJSON(config.info) && JSON.parse(config.info)) || {};
	const configTextStyle = configInfoJSON.textStyle || {}; // 文本的样式
	const configTextBreakLength = configInfoJSON.textBreakLength || undefined; // 字数过宽换行

	const width = config.wide ? config.wide + 'px' : 'auto';
	const height = config.high ? config.high + 'px' : 'auto';
	if (configTextBreakLength && text.length > configTextBreakLength) {
		text = [text.substr(0, configTextBreakLength), text.substr(configTextBreakLength)];
		return text.map((x, index) => {
			return (
				<div
					key={x}
					style={{
						...styleCount(config),
						marginBottom: '0',
						width,
						textAlign: 'right',
						...configTextStyle,
					}}>
					{x}
				</div>
			);
		});
	}
	if (config.mark && text.indexOf(config.mark) > -1) {
		// 换行分隔符
		text = text.split(config.mark);
		let length = text.length;
		return text.map((x, index) => {
			return (
				<div
					key={x}
					style={{
						...styleCount(config),
						marginBottom: '0',
						width,
						textAlign: 'right',
						...configTextStyle,
					}}>
					{x}
					{length > index + 1 ? config.mark : ''}
				</div>
			);
		});
	}

	return (
		<div
			style={{
				...styleCount(config),
				width,
				height,
				lineHeight: 'normal',
				verticalAlign: 'top',
				textAlign: 'right',
				letterSpacing: '2px',
				...configTextStyle,
			}}>
			{text}
		</div>
	);
};

const imageRender = (imageUrl) => {
	return (
		<img
			src={imageUrl} //  TODO 这里图片样式需要考虑怎么兼容所有业务
			style={{
				height: '100%',
				width: 'auto',
			}}
		/>
	);
};

class BarCode extends React.Component {
	toJsBarcode() {
		// 调用 JsBarcode方法生成条形码
		const { value } = this.props;
		let info = value;
		if (/.*[\u4e00-\u9fa5]+.*$/.test(value)) {
			info = 'error';
			console.log('条形码包含中文不能输出', value);
		}
		JsBarcode(this.barcode, info, {
			text: '',
			format: 'CODE128',
			displayValue: false,
			width: 1.0,
			height: 30,
			margin: 0,
		});
		return true;
	}

	componentDidMount() {
		this.toJsBarcode();
	}

	render() {
		return (
			<div className="barcode-box">
				<svg
					ref={(ref) => {
						this.barcode = ref;
					}}
				/>
			</div>
		);
	}
}

export default function DetailDom({ config, data }) {
	const { x, y, transform } = config;
	const styleOutSide = {
		position: 'absolute',
		zIndex: 10,
		fontWeight: '600',
		left: x + 'px',
		top: y + 'px',
	};
	if (transform) {
		styleOutSide.transform = `rotate(${transform}deg)`;
		styleOutSide.WebkitTransform = `rotate(${transform}deg)`;
		styleOutSide.MozTransform = `rotate(${transform}deg)`;
		styleOutSide.MsTransform = `rotate(${transform}deg)`;
		//    styleOutSide.MsFilter = "progid:DXImageTransform.Microsoft.Matrix(M11=-1, M12=1.2246467991473532e-16, M21=-1.2246467991473532e-16, M22=-1, SizingMethod='auto expand')";
	}

	const outSideDom = (children) => {
		return (
			<div
				style={{
					...styleOutSide,
				}}
				key={`filed${config.id}`}>
				{children}
			</div>
		);
	};
	let children = null;
	switch (config.fieldType) {
		case '-2':
			children = <BarCode value={data} />;
			break;
		case '-1': // 常量
			children = normalTextRender(config.content, config);
			break;
		case '0': // 文字 带参数
			let text = data;
			const strPlace = config.content && config.content.indexOf('${');
			if (strPlace > -1) {
				const str = `\${${config.fieldCode}}`;
				text = config.content.replace(str, text);
			}
			children = normalTextRender(text, config);
			break;
		case '1': // 时间
			let time = data;
			time = moment(time).format(config.fieldPattern || 'YYYY-MM-DD');
			children = normalTextRender(time, config);
			break;
		case '2': // 照片 图章
			children = imageRender(data);
			break;
		default:
			console.error('未知的打印数据类型,无法渲染');
			children = null;
			break;
	}
	return outSideDom(children);
}
