import React, { useState, useEffect } from 'react';
import { Form, Button } from 'antd';
import Shell from '@/baseComponent/Shell';
import { ModalInfo } from '@/baseComponent/Modal';
import { addOrEditTableItem, fetchTableData, fetchTableHeader } from '@/webPublic/Services';
import { preHandle } from '@/webPublic/one_stop_public/utils/myutils';
import { getOneStopConfig } from '@/baseComponent/utils';
import { RenderFormByContent } from '../RenderForm';

// 直接获取的表格数据的字段是明文，找换成base转换后的字段
function getBaseEncodedData(rowData, tableHeader) {
	// {name: base52}
	const nameBaseMap = tableHeader.reduce((acc, header) => {
		return { ...acc, [header.name]: header.base52 };
	}, {});
	// 明文字段转换成base后的字段, eg. {process_status: 1} => {JbZWZHoGRsA: 1}
	return Object.keys(rowData).reduce((acc, name) => {
		const base52 = nameBaseMap[name];
		if (!base52) return acc;
		return { ...acc, [base52]: rowData[name] };
	}, {});
}

// 配置页面，配置数据在元数据表格的第一行
// 1. 页面显示通过模板id渲染。
// 2. 页面数据通过元数据表id, 查询和修改。
// templateCode, 页面模板id
// tableId, 元数据列表id
// children， 可以传children，但children不能是数组(不能传属性)，children里可以自定义其他组件。

function SingleDataPageInFirstRow(props) {
	const {
		children,
		form,
		templateCode: templateCodeFromProps,
		dataObjId: dataObjIdFromProps,
		location: { pathname },
	} = props;
	// templateCode 和 dataObjId 可以通过组件属性或者根据路径配置获取
	const templateCodeFromPathname = getOneStopConfig(`${pathname}-templateCode`);
	const dataObjIdFromPathname = getOneStopConfig(`${pathname}-dataObjId`);

	const templateCode = templateCodeFromPathname || templateCodeFromProps;
	const dataObjId = dataObjIdFromPathname || dataObjIdFromProps;

	if (!templateCode || !dataObjId) {
		console.warn(
			'模板code或者元数据表格id为空，templateCode: ' + templateCode + ', dataObjId: ' + dataObjId,
		);
		return null;
	}

	const [isAdd, setIsAdd] = useState(false);
	const [content, setContent] = useState({});

	useEffect(
		() => {
			fetchTableHeader(dataObjId).then((tableHeader) => {
				if (!tableHeader) return;
				fetchTableData({ pageNo: 1, pageSize: 10, isBase: true, dataObjId, query: {} }).then(
					(res) => {
						const firstRowData = (res && res.rows && res.rows[0]) || {};
						setContent(getBaseEncodedData(firstRowData, tableHeader));
						setIsAdd(!!(!res || res.errMsg));
					},
				);
			});
		},
		[dataObjId],
	);

	function handleSave() {
		const { validateFields } = form;

		validateFields((err, values) => {
			if (err) return;
			// console.log(JSON.stringify(values.JjvkRobXWTE), JSON.stringify(values.JjvkwLqcsyY));
			preHandle(values);
			// console.log(JSON.stringify(values.JjvkRobXWTE));
			addOrEditTableItem({ objId: dataObjId, isAdd, data: values }).then((res) => {
				if (res) {
					ModalInfo('保存成功！');
				}
			});
		});
	}

	let ClonedChildren;
	if (children) {
		ClonedChildren = React.cloneElement(React.Children.only(children), {
			form,
			isAdd,
			url: '/DataObjApi/addFormData',
		});
	}

	return (
		<Shell styleShell={{ marginTop: 0 }}>
			<RenderFormByContent content={content} templateCode={templateCode} form={form} />
			{ClonedChildren || (
				<div style={{ textAlign: 'center', padding: 16 }}>
					<Button type="primary" shape="round" ghost onClick={handleSave}>
						保存
					</Button>
				</div>
			)}
		</Shell>
	);
}

export default Form.create()(SingleDataPageInFirstRow);
