import assocIndexOf from './assocIndexOf.js';

class ListCache {
	/**
	 * Creates an list cache object.
	 * 创建一个列表缓存对象。
	 * @private
	 * @constructor
	 * @param {Array} [entries] 要缓存的键值对。
	 */
	constructor(entries) {
		let index = -1;
		const length = entries == null ? 0 : entries.length;

		this.clear();
		while (++index < length) {
			const entry = entries[index];
			this.set(entry[0], entry[1]);
		}
	}

	/**
	 * Removes all key-value entries from the list cache.
	 * 从列表缓存中删除所有键值项。
	 * @memberOf ListCache
	 */
	clear() {
		this.__data__ = [];
		this.size = 0;
	}

	/**
	 * Removes `key` and its value from the list cache.
	 * 从列表缓存中删除' key '及其值。
	 * @memberOf ListCache
	 * @param {string} key 要删除的值的键。
	 * @returns {boolean} 如果条目被删除，则返回“true”，否则返回“false”。
	 */
	delete(key) {
		const data = this.__data__;
		const index = assocIndexOf(data, key);

		if (index < 0) {
			return false;
		}
		const lastIndex = data.length - 1;
		if (index == lastIndex) {
			data.pop();
		} else {
			data.splice(index, 1);
		}
		--this.size;
		return true;
	}

	/**
	 * Gets the list cache value for `key`.
	 * 获取' key '的列表缓存值。
	 * @memberOf ListCache
	 * @param {string} key 要获取的值的键。
	 * @returns {*} 返回条目值。
	 */
	get(key) {
		const data = this.__data__;
		const index = assocIndexOf(data, key);
		return index < 0 ? undefined : data[index][1];
	}

	/**
	 * Checks if a list cache value for `key` exists.
	 * 检查' key '的列表缓存值是否存在。
	 * @memberOf ListCache
	 * @param {string} key 输入键要检查。
	 * @returns {boolean} 如果' key '的条目存在，则返回' true '，否则返回' false '。
	 */
	has(key) {
		return assocIndexOf(this.__data__, key) > -1;
	}

	/**
	 * Sets the list cache `key` to `value`.
	 * 将列表缓存“键”设置为“值”。
	 * @memberOf ListCache
	 * @param {string} key 要设置的值的键
	 * @param {*} value 要设置的值。
	 * @returns {Object} 返回列表缓存实例。
	 */
	set(key, value) {
		const data = this.__data__;
		const index = assocIndexOf(data, key);

		if (index < 0) {
			++this.size;
			data.push([key, value]);
		} else {
			data[index][1] = value;
		}
		return this;
	}
}

export default ListCache;
