import isArguments from './isArguments.js';
import isBuffer from './isBuffer.js';
import isIndex from './isIndex.js';
import isTypedArray from './isTypedArray.js';

/** Used to check objects for own properties. */
/** 用于检查对象的自身属性。 */
const hasOwnProperty = Object.prototype.hasOwnProperty;

/**
 * Creates an array of the enumerable property names of the array-like `value`.
 * 创建类似数组的“值”的可枚举属性名的数组。
 * @private
 * @param {*} value 要查询的值.
 * @param {boolean} inherited 指定返回继承的属性名.
 * @returns {Array} 返回属性名数组.
 */
function arrayLikeKeys(value, inherited) {
	const isArr = Array.isArray(value);
	const isArg = !isArr && isArguments(value);
	const isBuff = !isArr && !isArg && isBuffer(value);
	const isType = !isArr && !isArg && !isBuff && isTypedArray(value);
	const skipIndexes = isArr || isArg || isBuff || isType;
	const length = value.length;
	const result = new Array(skipIndexes ? length : 0);
	let index = skipIndexes ? -1 : length;
	while (++index < length) {
		result[index] = `${index}`;
	}
	for (const key in value) {
		if (
			(inherited || hasOwnProperty.call(value, key)) &&
			!(
				skipIndexes &&
				// Safari 9 has enumerable `arguments.length` in strict mode.
				(key === 'length' ||
					// Skip index properties.
					isIndex(key, length))
			)
		) {
			result.push(key);
		}
	}
	return result;
}

export default arrayLikeKeys;
