import React, { Component } from 'react';
import { Table } from 'antd';
import styles from './index.less';
import PropTypes from 'prop-types';

const giveNewColumns = (columns, tableId) => {
	let scrollX = 0;
	for (let item of columns) {
		if (item != null) {
			scrollX = typeof item.width !== 'undefined' ? scrollX + item.width : scrollX + 100;
		}
	}
	const width = document.getElementById(tableId).offsetWidth;
	if (width < scrollX) {
		if (columns.length) {
			let lastItem = columns[columns.length - 1];
			if (
				(lastItem.title === '操作' || lastItem.title === '详情') &&
				typeof lastItem.fixed === 'undefined' &&
				typeof lastItem.noNeedFixed === 'undefined'
			) {
				columns[columns.length - 1].fixed = 'right'; // 操作/ 详情列 增加 浮动
			}
		}
		return {
			scrollX,
			columns,
		};
	} else {
		return null;
	}
};
export default class NormalTable extends Component {
	constructor(props) {
		super(props);
		this.state = {
			tableId: Math.random()
				.toString(36)
				.substr(2),
			scrollX: 0,
			columns: props.columns,
		};
	}

	static getDerivedStateFromProps(nextProps, prevState) {
		if (JSON.stringify(nextProps.columns) === JSON.stringify(prevState.columns)) {
			return null;
		} else {
			const columns = nextProps.columns;
			let newState = giveNewColumns(columns, prevState.tableId);
			if (newState) {
				return {
					scrollX: newState.scrollX,
					columns: newState.columns,
				};
			} else {
				return {
					columns,
				};
			}
		}
	}

	/**
	 * <Table> dom 占得的宽度offsetWidth与计算所得的列宽scrollX
	 * 两者进行比较确定Table scroll的值
	 */
	componentDidMount() {
		const { tableId, columns } = this.state;
		let state = giveNewColumns(columns, tableId);
		if (state) {
			this.setState(
				{
					scrollX: state.scrollX,
				},
				() => {
					this.setState({
						columns: state.columns,
					});
				},
			);
		}
	}

	render() {
		const {
			dataSource,
			rowKey,
			handleSelectRow,
			pagination,
			components,
			loading,
			onChange,
			bordered,
			scroll,
			showHeader,
		} = this.props;
		const { columns } = this.state;

		/**
		 * 没有传rowSelection 传了handleSelectRow 时 使用 handleSelectRow 并封装rowSelection
		 * 传了rowSelection 直接使用
		 * */
		const rowSelection =
			handleSelectRow === false
				? null
				: {
						columnWidth: 30,
						fixed: true,
						hideDefaultSelections: false,
						onChange: (selectedRowKeys, selectedRows) => {
							handleSelectRow(selectedRowKeys, selectedRows);
						},
				  };
		const { scrollX } = this.state;
		if (pagination && pagination.total) {
			pagination.showTotal = (total, range) => `共${total}条 　`;
		}
		// console.log(rowKey);
		return (
			<Table
				dataSource={dataSource}
				showHeader={showHeader}
				columns={columns}
				components={components}
				rowKey={rowKey || 'key'}
				loading={loading}
				bordered={bordered}
				id={this.state.tableId}
				className={styles.NormalTable}
				pagination={pagination ? pagination : false}
				onChange={onChange}
				scroll={scrollX > 0 ? { x: scrollX } : scroll}
				rowSelection={
					this.props.rowSelection !== 'undefined' ? this.props.rowSelection : rowSelection
				}
			/>
		);
	}
}
NormalTable.propTypes = {
	dataSource: PropTypes.array.isRequired, // 表格数据源 To Antd Table
	bordered: PropTypes.bool, // 是否显示border,
	columns: PropTypes.array.isRequired, // 字段 To Antd Table
	rowKey: PropTypes.string, // rowKey To Antd Table
	handleSelectRow: PropTypes.oneOfType([PropTypes.func, PropTypes.bool]), // 行选择的事件,传false 表示不开放行选择 To Antd Table
	pagination: PropTypes.oneOfType([
		// 多种类型的数据
		PropTypes.object,
		PropTypes.bool,
	]),
	components: PropTypes.object, // 默认覆盖的 Table元素 基本没用
	loading: PropTypes.bool,
	onChange: PropTypes.func, // 分页、排序、筛选变化时触发
};
NormalTable.defaultProps = {
	rowKey: 'id',
	handleSelectRow: (selectedRowKeys, selectedRows) => {
		//console.log(selectedRowKeys, selectedRows);
	}, // false,
	pagination: {
		defaultCurrent: 1,
		total: 20,
		pageSize: 5,
		showQuickJumper: true,
		onChange: (current, size) => {
			//console.log(current, size);
		},
	},
	loading: false,
	bordered: false,
	components: undefined,
	onChange: () => {
		//console.log('NormalTable.onChange');
	},
	dataSource: [],
	columns: [
		{
			dataIndex: 'id',
			title: 'id',
		},
		{
			dataIndex: 'operation',
			title: '操作',
			fixed: 'right',
			width: 120,
			render: (text, record) => {
				return (
					<span>
						<a>编辑 </a>|<a> 删除</a>
					</span>
				);
			},
		},
	],
};
